from pyparsing import Literal, Word

from codeable_detectors.basic_detectors import AtLeastOneFileMatchesDetector
from codeable_detectors.detector_context import DetectorContext
from codeable_detectors.evidences import FailedEvidence, LinkEvidence
from codeable_detectors.java.java_detectors import JavaInvocation
from codeable_detectors.pyparsing_patterns import ID


def detect_js_import(ctx, package_name):
    matches = []
    require_matches = ctx.matches_pattern(ID + Literal("=") + Literal("require") +
                                          Literal("(") + Word("`'\"", max=1) + Literal(package_name) +
                                          Word("`'\"", max=1) + Literal(")"))
    for match in require_matches:
        import_var = match.text[:match.text.find("=")].strip()
        match.update_keyword_args(package_name=package_name, importVar=import_var)
    matches.extend(require_matches)
    return matches


class JSImportLink(AtLeastOneFileMatchesDetector):
    def __init__(self, package_name):
        super().__init__()
        self.file_endings = ["js"]
        self.package_name = package_name

    def detect_in_context(self, ctx, **kwargs):
        import_matches = detect_js_import(ctx, self.package_name)
        if not import_matches:
            return FailedEvidence("JS '" + str(self.package_name) + "' import not found")
        return LinkEvidence(import_matches).set_properties(kwargs=kwargs)


class JSInvocation(JavaInvocation):
    def __init__(self, invocation_name):
        super().__init__(invocation_name)
        self.file_endings = ["js"]


class JSCallInPromiseLinks(AtLeastOneFileMatchesDetector):
    def __init__(self, call_detector):
        super().__init__()
        self.promise_detector = JSInvocation("Promise")
        self.call_detector = call_detector
        self.file_endings = ["js"]

    def detect_in_context(self, ctx, **kwargs):
        promise_evidence = self.promise_detector.detect_in_context(ctx, **kwargs)
        if promise_evidence.has_failed():
            return promise_evidence

        link_evidence = None
        for promise_match in promise_evidence.matches:
            call_evidence = self.call_detector.detect_in_context(DetectorContext(promise_match), **kwargs)
            if not call_evidence.has_failed():
                if link_evidence is None:
                    link_evidence = call_evidence
                else:
                    link_evidence.update(call_evidence)
        if not link_evidence or not link_evidence.matches:
            return FailedEvidence("no JS call in a promise found")
        return link_evidence.set_properties(kwargs=kwargs)


# detect JS request links and check whether it is an async call via a promise, just a sync call (no promise)
# or a combination of both; other ways to make JS call async need to be added later.
class JSCallSyncAsyncDetector(AtLeastOneFileMatchesDetector):
    def __init__(self, call_detector):
        super().__init__()
        self.call_in_promise_detector = JSCallInPromiseLinks(call_detector)
        self.call_detector = call_detector
        self.file_endings = ["js"]

    def detect_in_context(self, ctx, **kwargs):
        in_promise_evidence = self.call_in_promise_detector.detect_in_context(ctx, **kwargs)
        call_evidence = self.call_detector.detect_in_context(ctx, **kwargs)

        if in_promise_evidence.has_succeeded():
            evidence = in_promise_evidence
            link_type = "asynchronousConnector"
            if call_evidence:
                if len(call_evidence.matches) > len(in_promise_evidence.matches):
                    # some more sync links, in addition to the async ones
                    link_type = "syncAsyncConnector"
        elif call_evidence.has_succeeded():
            evidence = call_evidence
            link_type = "synchronousConnector"
        else:
            return FailedEvidence("no JS call found")

        evidence.link_types.append(link_type)
        return evidence
